<?php
if (!defined('ABSPATH')) exit;

class KLB_Social_Login {

    const VERSION = '1.1.0';
    const QUERY_FLAG = 'klb_oauth';
    const STATE_TTL  = 10 * MINUTE_IN_SECONDS;

    public static function boot() {
        add_action('init', [__CLASS__, 'add_rewrite_and_vars']);
        add_action('template_redirect', [__CLASS__, 'maybe_handle_oauth']);
        add_action('wp_enqueue_scripts', [__CLASS__, 'enqueue_assets']);

        if (!self::is_enabled()) return;

        $loc = (array) get_theme_mod('klb_oauth_locations', ['login','register']);

        if (in_array('login', $loc, true)) {
            add_action('woocommerce_login_form_end', [__CLASS__, 'buttons']);
        }
        if (in_array('register', $loc, true)) {
            add_action('woocommerce_register_form_end', [__CLASS__, 'buttons']);
        }

        add_shortcode('klb_social_login', [__CLASS__, 'shortcode']);
    }

    /** ---------------- Settings helpers ---------------- */

    private static function is_enabled() {
        return (bool) get_theme_mod('klb_oauth_enabled', true);
    }
    private static function provider_on($p) {
        if ($p === 'google')   return (bool) get_theme_mod('klb_oauth_google_on', true);
        if ($p === 'facebook') return (bool) get_theme_mod('klb_oauth_facebook_on', true);
        return false;
    }
    private static function cred($key) {
        // Retrieves theme_mod values; returns '' if missing
        return (string) get_theme_mod($key, '');
    }
    private static function allowed_redirect_slugs() {
        $rows = (array) get_theme_mod('klb_oauth_allowed_redirects', []);
        $slugs = [];
        foreach ($rows as $row) {
            if (is_array($row) && !empty($row['slug'])) {
                $slugs[] = sanitize_title_with_dashes($row['slug']);
            } elseif (is_string($row)) {
                $slugs[] = sanitize_title_with_dashes($row);
            }
        }
        // Always ensure 'home' exists
        if (!in_array('home', $slugs, true)) $slugs[] = 'home';
        return array_unique($slugs);
    }
    private static function label($provider) {
        if ($provider === 'google') {
            $d = get_theme_mod('klb_oauth_label_google', 'Google');
            return $d ?: 'Google';
        }
        if ($provider === 'facebook') {
            $d = get_theme_mod('klb_oauth_label_facebook', 'Facebook');
            return $d ?: 'Facebook';
        }
        return ucfirst($provider);
    }

    /** ---------------- UI ---------------- */

    public static function buttons() {
        if (is_user_logged_in() || !self::is_enabled()) return;
        $redirect = self::get_safe_redirect();

        $has_one = self::provider_on('google') || self::provider_on('facebook');
        if (!$has_one) return;

        echo '<div class="klb-social-login">';
		echo '<p class="social-divider"><span>'.esc_html__('Or login with', 'cosmetsy-core').'</span></p>';
        if (self::provider_on('google') && self::cred('klb_oauth_google_client_id')) {
            echo self::provider_button('google', $redirect);
        }
        if (self::provider_on('facebook') && self::cred('klb_oauth_facebook_client_id')) {
            echo self::provider_button('facebook', $redirect);
        }
        echo '</div>';
    }

    public static function shortcode($atts = []) {
        ob_start();
        self::buttons();
        return ob_get_clean();
    }

    private static function provider_button($provider, $redirect) {
        $url   = esc_url(self::build_auth_url($provider, $redirect));
        $label = esc_html(self::label($provider));
        $class = 'klb-social-login-btn klb-btn-' . esc_attr($provider);
        return "<a class='{$class}' href='{$url}' rel='nofollow noopener'>{$label}</a>";
    }

    public static function enqueue_assets() {
		wp_enqueue_style( 'klb-social-login', plugins_url( 'css/social-login.css', __FILE__ ), true );
    }

    /** --------------- Core --------------- */

    public static function add_rewrite_and_vars() {
        add_rewrite_tag('%'.self::QUERY_FLAG.'%', '([^&]+)');
        add_rewrite_tag('%provider%', '([^&]+)');
        // Pretty URLs (optional): uncomment if you want /oauth/callback/{provider}
        // add_rewrite_rule('^oauth/callback/([^/]+)/?', 'index.php?'.self::QUERY_FLAG.'=callback&provider=$matches[1]', 'top');
    }

    public static function maybe_handle_oauth() {
        $action   = get_query_var(self::QUERY_FLAG) ?: (isset($_GET[self::QUERY_FLAG]) ? sanitize_text_field($_GET[self::QUERY_FLAG]) : '');
        $provider = get_query_var('provider') ?: (isset($_GET['provider']) ? sanitize_text_field($_GET['provider']) : '');

        if (empty($action) || empty($provider) || !self::is_enabled() || !self::provider_on($provider)) return;

        if ($action === 'auth') {
            $redirect = self::get_safe_redirect();
            wp_redirect(self::build_provider_auth_link($provider, $redirect));
            exit;
        }

        if ($action === 'callback') {
            self::handle_callback($provider);
            exit;
        }
    }

    /** --------------- OAuth helpers --------------- */

    private static function build_auth_url($provider, $redirect) {
        $args = [
            self::QUERY_FLAG => 'auth',
            'provider'       => $provider,
            'redirect_to'    => rawurlencode($redirect),
        ];
        return add_query_arg($args, home_url('/'));
    }

    private static function callback_url($provider) {
        return add_query_arg([
            self::QUERY_FLAG => 'callback',
            'provider'       => $provider,
        ], home_url('/'));
    }

    private static function build_provider_auth_link($provider, $redirect) {
        $state = wp_generate_password(24, false);
        set_transient('klb_oauth_state_' . $state, [
            't'        => time(),
            'provider' => $provider,
            'redirect' => $redirect,
        ], self::STATE_TTL);

        if ($provider === 'google') {
            $client = self::cred('klb_oauth_google_client_id');
            $params = [
                'client_id'     => $client,
                'redirect_uri'  => self::callback_url('google'),
                'response_type' => 'code',
                'scope'         => 'openid email profile',
                'state'         => $state,
                'access_type'   => 'online',
                'prompt'        => 'select_account',
            ];
            return 'https://accounts.google.com/o/oauth2/v2/auth?' . http_build_query($params);
        }

        if ($provider === 'facebook') {
            $client = self::cred('klb_oauth_facebook_client_id');
            $params = [
                'client_id'     => $client,
                'redirect_uri'  => self::callback_url('facebook'),
                'response_type' => 'code',
                'scope'         => 'email,public_profile',
                'state'         => $state,
                'auth_type'     => 'rerequest',
            ];
            return 'https://www.facebook.com/v19.0/dialog/oauth?' . http_build_query($params);
        }

        wp_die('Unsupported provider.');
    }

    private static function handle_callback($provider) {
        $err = isset($_GET['error']) ? sanitize_text_field($_GET['error']) : '';
        if ($err) wp_die('OAuth error: ' . esc_html($err));

        $code  = isset($_GET['code']) ? sanitize_text_field($_GET['code']) : '';
        $state = isset($_GET['state']) ? sanitize_text_field($_GET['state']) : '';

        $saved = get_transient('klb_oauth_state_' . $state);
        delete_transient('klb_oauth_state_' . $state);
        if (!$saved || empty($code) || $saved['provider'] !== $provider) {
            wp_die('Invalid OAuth state.');
        }

        $token = self::exchange_token($provider, $code);
        if (is_wp_error($token)) wp_die($token);

        $u = self::fetch_userinfo($provider, $token);
        if (is_wp_error($u)) wp_die($u);

        $user_id = self::upsert_user($provider, $u);
        if (is_wp_error($user_id)) wp_die($user_id);

        wp_set_auth_cookie($user_id, true);
        do_action('klb_social_login_success', $user_id, $provider, $u);

        $redir = self::sanitize_redirect_target($saved['redirect']);
        wp_safe_redirect($redir ? $redir : (function_exists('wc_get_page_permalink') ? wc_get_page_permalink('myaccount') : home_url('/')));
        exit;
    }

    private static function exchange_token($provider, $code) {
        if ($provider === 'google') {
            $resp = wp_remote_post('https://oauth2.googleapis.com/token', [
                'timeout' => 20,
                'body'    => [
                    'code'          => $code,
                    'client_id'     => self::cred('klb_oauth_google_client_id'),
                    'client_secret' => self::cred('klb_oauth_google_client_secret'),
                    'redirect_uri'  => self::callback_url('google'),
                    'grant_type'    => 'authorization_code',
                ],
            ]);
        } elseif ($provider === 'facebook') {
            $resp = wp_remote_get(add_query_arg([
                'client_id'     => self::cred('klb_oauth_facebook_client_id'),
                'client_secret' => self::cred('klb_oauth_facebook_client_secret'),
                'redirect_uri'  => self::callback_url('facebook'),
                'code'          => $code,
            ], 'https://graph.facebook.com/v19.0/oauth/access_token'), ['timeout' => 20]);
        } else {
            return new WP_Error('oauth_provider', 'Unsupported provider');
        }

        if (is_wp_error($resp)) return $resp;
        $code_http = wp_remote_retrieve_response_code($resp);
        $body = json_decode(wp_remote_retrieve_body($resp), true);
        if ($code_http >= 400 || !is_array($body)) {
            return new WP_Error('oauth_token', 'Token exchange failed');
        }
        return $body;
    }

    private static function fetch_userinfo($provider, $token) {
        if ($provider === 'google') {
            $resp = wp_remote_get('https://www.googleapis.com/oauth2/v3/userinfo', [
                'timeout' => 20,
                'headers' => ['Authorization' => 'Bearer ' . $token['access_token']],
            ]);
            if (is_wp_error($resp)) return $resp;
            $data = json_decode(wp_remote_retrieve_body($resp), true);
            if (empty($data['email'])) return new WP_Error('oauth_email', 'No email returned from Google');
            return [
                'id'        => $data['sub'] ?? '',
                'email'     => sanitize_email($data['email']),
                'name'      => sanitize_text_field($data['name'] ?? ''),
                'first'     => sanitize_text_field($data['given_name'] ?? ''),
                'last'      => sanitize_text_field($data['family_name'] ?? ''),
                'avatar'    => esc_url_raw($data['picture'] ?? ''),
                'verified'  => !empty($data['email_verified']),
            ];
        }

        if ($provider === 'facebook') {
            $resp = wp_remote_get(add_query_arg([
                'fields'       => 'id,name,email,first_name,last_name,picture.type(large)',
                'access_token' => $token['access_token'] ?? '',
            ], 'https://graph.facebook.com/v19.0/me'), ['timeout' => 20]);
            if (is_wp_error($resp)) return $resp;
            $data = json_decode(wp_remote_retrieve_body($resp), true);
            if (empty($data['email'])) {
                return new WP_Error('oauth_email', 'No email returned from Facebook (permissions/app review).');
            }
            return [
                'id'        => $data['id'] ?? '',
                'email'     => sanitize_email($data['email']),
                'name'      => sanitize_text_field($data['name'] ?? ''),
                'first'     => sanitize_text_field($data['first_name'] ?? ''),
                'last'      => sanitize_text_field($data['last_name'] ?? ''),
                'avatar'    => esc_url_raw($data['picture']['data']['url'] ?? ''),
                'verified'  => true,
            ];
        }

        return new WP_Error('oauth_provider', 'Unsupported provider');
    }

    private static function upsert_user($provider, $u) {
        $meta_key = 'klb_'.$provider.'_id';

        $existing = get_users([
            'meta_key'   => $meta_key,
            'meta_value' => $u['id'],
            'number'     => 1,
            'fields'     => 'ID',
        ]);
        if (!empty($existing)) {
            $user_id = (int) $existing[0];
            self::maybe_update_avatar($user_id, $u['avatar']);
            return $user_id;
        }

        $user = get_user_by('email', $u['email']);
        if ($user) {
            update_user_meta($user->ID, $meta_key, $u['id']);
            self::maybe_update_profile_names($user->ID, $u);
            self::maybe_update_avatar($user->ID, $u['avatar']);
            return $user->ID;
        }

        $username = sanitize_user(current(explode('@', $u['email'])));
        if (username_exists($username)) {
            $username .= '_' . wp_generate_password(6, false, false);
        }

        $random_password = wp_generate_password(24);
        $user_id = wp_create_user($username, $random_password, $u['email']);
        if (is_wp_error($user_id)) return $user_id;

        wp_update_user([
            'ID'           => $user_id,
            'first_name'   => $u['first'] ?? '',
            'last_name'    => $u['last'] ?? '',
            'display_name' => $u['name'] ?: $username,
        ]);

        update_user_meta($user_id, $meta_key, $u['id']);
        update_user_meta($user_id, 'klb_social_provider', $provider);
        update_user_meta($user_id, 'klb_social_verified_email', !empty($u['verified']) ? '1' : '0');
        self::maybe_update_avatar($user_id, $u['avatar']);

        // Default WooCommerce role
        $user_obj = new WP_User($user_id);
        if (empty(array_intersect(['customer','subscriber','shop_manager','administrator'], $user_obj->roles))) {
            $user_obj->set_role('customer');
        }

        return $user_id;
    }

    private static function maybe_update_profile_names($user_id, $u) {
        $needs = [];
        if (!empty($u['first']) && !get_user_meta($user_id, 'first_name', true)) $needs['first_name'] = $u['first'];
        if (!empty($u['last'])  && !get_user_meta($user_id, 'last_name', true))  $needs['last_name']  = $u['last'];
        if (!empty($needs)) {
            $needs['ID'] = $user_id;
            wp_update_user($needs);
        }
    }

    private static function maybe_update_avatar($user_id, $avatar_url) {
        if (!$avatar_url) return;
        update_user_meta($user_id, 'klb_social_avatar', esc_url_raw($avatar_url));
    }

    /** --------------- Current URL --------------- */

	private static function current_url() {
	    $scheme = is_ssl() ? 'https://' : 'http://';
	    $host   = $_SERVER['HTTP_HOST'] ?? '';
	    $uri    = $_SERVER['REQUEST_URI'] ?? '/';
	    // #anchor' at, gereksiz oauth paramlarn kaldr
	    $raw    = $scheme . $host . strtok($uri, '#');
	    return remove_query_arg(
	        [ self::QUERY_FLAG, 'provider', 'code', 'state', 'error', 'redirect_to' ],
	        $raw
	    );
	}

    /** --------------- Redirect safety --------------- */

	private static function get_safe_redirect() {
	    $home     = home_url('/');
	    $fallback = function_exists('wc_get_page_permalink') ? wc_get_page_permalink('myaccount') : $home;
	
	    // 1) If explicit redirect_to
	    if (isset($_REQUEST['redirect_to'])) {
	        $raw = rawurldecode(sanitize_text_field($_REQUEST['redirect_to']));
	        return self::sanitize_redirect_target($raw) ?: $fallback;
	    }
	
	    // 2) if checkout, checkout. if is my-account, my-account
	    if (function_exists('is_checkout') && is_checkout()) {
	        return self::sanitize_redirect_target(wc_get_checkout_url()) ?: $fallback;
	    }
	    if (function_exists('is_account_page') && is_account_page()) {
	        return self::sanitize_redirect_target(wc_get_page_permalink('myaccount')) ?: $fallback;
	    }
	
	    $current = self::current_url();
	    if ($current) {
	        return self::sanitize_redirect_target($current) ?: $fallback;
	    }

	    $ref = wp_get_referer();
	    return self::sanitize_redirect_target($ref ?: $fallback) ?: $fallback;
	}
	
	private static function sanitize_redirect_target($url) {
	    $home = home_url('/');
	    // only check same-origin, skip slug whitelist
	    return (strpos($url, $home) === 0) ? $url : $home;
	}
}

KLB_Social_Login::boot();
